/*
 * (C) 2003 henk kloepping <henk@opensource.nl>
 * 
 * $Id: nds.c,v 1.2 2003/12/22 20:54:57 hkloepping Exp $
 *
 * This program authenticates a user against PAM. If no user was specified 
 * on the command line it uses the name of the current user. If the user 
 * could be authenticated (e.g. by typing his/her password, inserting the 
 * proper key) and a program (and its optional parameters) was specified it 
 * fires up the program as specified on the command line. If no program was 
 * specified, it will simply return a message "Authenticated" or "Not 
 * Authenticated", unless the --silent option was specified, in which case 
 * it will simply return the proper status code.
 *
 * options: [ (-u|--user) user ] 
 *          [ (-s|--silent) | (-e|--exec) 'command and parameters' ]
 *
 * ---------------------------------------------------------------------------
 * Example PAM configuration for ldap authentication, file /etc/pam.d/nds: 
 * --
 * #%PAM-1.0
 * auth required pam_ldap.so debug
 * --
 *
 * Do not forget to set the proper values in /etc/openldap/ldap.conf, e.g.:
 *
 * --
 * host 10.1.1.72
 * base o=testcom,c=nl
 * ssl on
 * pam_password nds
 * --
 *
 * ---------------------------------------------------------------------------
 * Example PAM configuration for local password authentication, file 
 * /etc/pam.d/nds: 
 *
 * --
 * #%PAM-1.0
 * auth requisite pam_unix2.so
 * --
 *
 * NOTICE: if you want to authenticate against your local database using
 *         pam_unix2.so, your program needs read access for /etc/shadow. 
 *         This requires the nds programme to run setuid root (s-rwsr-sr-x)!
 *
 *         # chown root.nobody nds
 *         # chmod +s nds 
 *
 * ---------------------------------------------------------------------------
 * Example usage with ssh:  
 *
 * - paste public key for a user in authorized_keys, but restrict the 
 *   user to usage of a a command, e.g.:
 *
 *  --
 *  command="/fs0/testcom/bin/nds --user test --exec ls",no-port-forwarding, \
 *     no-X11-forwarding,no-agent-forwarding ssh-dss AAAAB3NzaC1kc3 ...
 *  --
 *
 * Now, when the user connects with the matching privkey the command
 * nds will be executed, which will try to authenticate the user using
 * the user credentials of 'test' and if succesful it will execute the
 * command 'ls'. The other options prevent forwarding.
 *
 * Another example:
 * --
 *  command="/fs0/testcom/bin/nds -exec ls",no-port-forwarding, \
 *     no-X11-forwarding,no-agent-forwarding ssh-dss AAAAB3NzaC1kc3 ...
 * --
 *
 * When the user connects with the matching privkey the command
 * nds will be executed, which will try to authenticate the user using
 * the user credentials he provided to ssh and if succesfull will execute 
 * the command 'ls'. The other options prevent forwarding.
 *
 * ---------------------------------------------------------------------------
 */

/* generic support
*/
#include <stdio.h>              /* for printf */
#include <stdlib.h>             /* for exit */
#include <getopt.h>             /* for option parsing etc. */
#include <string.h>             /* strdup et all */
#include <stdarg.h>             /* varargs etc.*/ 
#include <sys/types.h>          /* pwd et al */
#include <pwd.h>                /* pwd et al */
#include <syslog.h>             /* system message logger */
#include <security/pam_appl.h>  /* pam stuff */
#include <security/pam_misc.h>  /* pam stuff */

/* <header> ------------------------------------------------------*/

/* 
 * The structure for options 
 */
typedef struct opts {
	int silent;
	char **exec;
        char *exec_txt;
	char *user;
} OPTIONS;

/*
 * Necessary for PAM
 */
static struct pam_conv conv = {
   misc_conv,
   NULL
};

/*
 * function definitions 
 */
char **tpa(char *cmd, OPTIONS *options);
void get_them(int argc, char **argv, OPTIONS *options);
void fail(int exitcode, char *fmt, ...);

/* <header/> -----------------------------------------------------*/

int main(int argc, char *argv[]) {

	OPTIONS options = { 0, NULL, NULL, NULL };
        pam_handle_t *pamh=NULL;
        int retval;

        /* parse options and set defaults
        */
	get_them(argc,argv,&options);

        /* check if user is authenticated
        */
        retval = pam_start("nds", options.user, &conv, &pamh);
        if (retval == PAM_SUCCESS) retval = pam_authenticate(pamh, 0);

        if (pam_end(pamh,retval) != PAM_SUCCESS) {     /* close Linux-PAM */
                pamh = NULL;
                fail(-8,"nds: failed to release authenticator\n");
        }

        /* This is where we have been authorized or not, see PAM_SUCCESS. */

        /* Log the attempt. 
        */
        syslog(
           LOG_NOTICE,
           "could%s authenticate %s\n",
           retval==PAM_SUCCESS ? "" : " NOT",
           options.user
        );

        /* Done if we were started in silent mode
        */
        if (options.silent) return(retval==PAM_SUCCESS ? 0:1);

        if (retval == PAM_SUCCESS) {
            if (options.exec) {
                syslog(
                    LOG_NOTICE,
                    "accepted request to exec '%s' for user '%s'\n", 
                    options.exec_txt, 
                    options.user
                );

                /* Won't come back (well, it should not..)
                */
	        execvp(strdup(options.exec[0]),options.exec);
                fail(-100,"PANIC: execvp returned?");
            }
            fprintf(stdout, "Authenticated\n");
        } else {
            fprintf(stdout, "Not Authenticated\n");
        }
	exit(0);
}

void fail(int exitcode, char *fmt, ...) {
     va_list ap;
     va_start(ap, fmt);
     fprintf(stderr,"Failed: ");
     vfprintf(stderr,fmt,ap);
     fprintf(stderr,"\n");
     va_end(ap);
     exit(exitcode);
}

void get_them(int argc, char **argv, OPTIONS *o) 
{
   int c;
   char **nargv;
   struct passwd *pwd_entry;

   while (1) {
       int option_index = 0;
       static struct option long_options[] = {
                   {"silent", 0, 0, 's' },
                   {"exec",   1, 0, 'e' },
                   {"user",   1, 0, 'u' },
                   {0, 0, 0, 0 }
       };

       c = getopt_long (argc, argv, "su:e:", long_options, &option_index);
       if (c == -1) break;

       switch (c) {

               case 's':
                   if (o->silent) fail(-1,"superfluous --silent option");
		   o->silent=1;
                   break;

               case 'u':
                   if (o->user) fail(-2,"superfluous --user option");
                   o->user=strdup(optarg);
                   break;

               case 'e':
                   if (o->exec) fail(-3,"superfluous --exec option");
		   nargv=tpa(optarg,o);
                   break;

               case '?':
		   fail(-4,"option parsing");
                   break;

               default:
                   fail(-125,"PANIC: getopt returned character code 0%o\n", c);
               }
           }
           if (optind < argc) fail(-124,"superfluous CLI arguments");

           /* 
            * logic for arguments:
            *
            * if --silent is specified, you should not specify a --exec
            */
            if (o->silent && o->exec) fail(-5,"silent and exec are mutex");

           /*
            * if --user is NOT specified, return current user
            */
            if (!o->user) {
                pwd_entry=getpwuid(getuid());
                if (pwd_entry==NULL) fail(-123,"PANIC: can't get my username"); 
                o->user=strdup(pwd_entry->pw_name);
            }
}

/*
 * Convert a whitespace separated command string to a command array (which 
 * comes in handy for exec..) and return the array. 
 */
char **tpa(char *cmd, OPTIONS *o)
{
	int n;
	char *token,*b;

        o->exec_txt=strdup(cmd);

	/* count how many args there are  
	*/
	n=0;
	b=(char *)strdup(cmd);
	if (b==(char *)NULL) fail(-6,"strdup() failed - no memory?\n");

	token=(char *)strtok(b,"\t ");
	while(token!=NULL)
	{
		n++;

		token=(char *)strtok(NULL,"\t ");
	}
	free(b); 

	if (n<1) fail(-7,"no valid command to execute\n");

	/* construct the argv array 
	*/
	o->exec=(char **)calloc(1, (n+1) * sizeof(char *) );
	if (o->exec==(char **)NULL) fail(-9,"could not allocate memory\n");
	
	/* do it one more time
	*/
	n=0;
	b=(char *)strdup(cmd);
	if (b==(char *)NULL) fail(-10,"strdup() failed - no memory?\n");

	token=(char *)strtok(b," \t");
	while(token!=NULL)
	{
		o->exec[n]=token;
		n++;
		token=(char *)strtok(NULL,"\t ");
	}
	o->exec[n]=(char *)NULL;

	return(o->exec);

}
