/* $Id: dbxurl.c,v 1.2 2002/03/02 15:13:35 henk Exp $ 
 *
 * (C) 2002 and beyond - Henk Kloepping <henk@opensource.nl>
 *
 * This file contains the main() function and the functions that
 * determine the main flow within dbxurl.
 */
#include <termios.h>  /* to set modes for stdin */
#include <unistd.h>
#include <stdlib.h>
#include "dbxurl.h"

/* gateway to heaven
*/
int main(int argc, char *const *argv)
{
    int soptind, err=0;

    /* 
     * First, parse the options. If the users uses the wrong
     * syntax or asks for the version number or help, this function   
     * will not return.
     */
    options=get_options(argc, argv);

    /* 
     * If we came this far, the user made no obvious mistakes and 
     * probably wants us to process a stream (or streams). Options 
     * were set, either by default or by the user. optind was set
     * by getopt(3) and is the index in argv of the first 
     * argv-element that is not an option. If the value in argc is 
     * bigger than that in optind the user specified filenames.
     *
     * We will loop through these filenames /twice/: the first loop 
     * checks the readability/accessability of the files, the 
     * second loop performs the actual parsing of the files. The first 
     * loop is there to assures that errors in the specification of
     * filenames (typos, reference to files that we are not
     * allowed to read, etc.) are detected /before/ any output is 
     * generated. To be able to loop twice we have to save the value 
     * of the optind counter:
     */
    soptind=optind;

    /* The first loop:
    */
    while (optind < argc)
    {
        FILE *dummy;

        /*
         * Note: TODO: recognize '-' as a placeholder for stdin (in which
         *             case we have to prepend a file named '-' with a
         *             pathname, e.g. './-')
         */
        dummy=fopen(argv[optind],"r");
	if (dummy==(FILE *)NULL) {
                printf("Error: can not read %s\n",argv[optind]);
                err++;
	} else fclose(dummy);
        optind++;
    } 
    if (err>0) {
	fprintf(stderr,"%d files could not be opened for reading\n",err);
        exit(2);
    }

    /* For the second loop, we first need to restore the optind counter:
    */
    optind=soptind;

    /* If no files were specified (optind==argc) we go to demo mode,
     * which reads from stdin:
    */
    if (optind==argc) {
            process_stdin(stdout);
    }

    /* 
     * The second loop actually performs the filechecks again: the files
     * could have been removed, renamed or had their permissions changed
     * by another process. 
     */
    while (optind < argc)
    {
        FILE *dummy;

        dummy=fopen(argv[optind],"r");
	if (dummy!=(FILE *)NULL) { 

            /* 
             * A file was found, it can be read and now we process it:
             */
            process_file(dummy, stdout);
	    fclose(dummy);
	}
        optind++;
    } 

    /* 
     * Done. Return cleanly.  
     */
    exit(0);
}

/*
 * This function implements the 'stdin' mode, which reads input from 
 * standard input. 'stdin mode' is invoked when either the user did 
 * not specify any filenames or the program reads from a pipe. Note: if 
 * working from the console, 'control-D' will not be recognised as 
 * 'terminate input', but as 'a byte with value 4'.  
 */
void process_stdin(FILE *out)
{
    struct termios old, new;

    /* set raw io mode 
    */
    tcgetattr(0, &old);                        /* get settings for stdin */
    memcpy(&new,&old,sizeof(struct termios));  /* copy them over */
    new.c_lflag &= ~(ECHO|ECHONL|ICANON);      /* configure raw input */
    tcsetattr(0, TCSANOW, &new);               /* set it */

    process_file(stdin, out);

    /* set cooked again
    */
    tcsetattr(0, TCSANOW, &old);
}

/* 
 * The function that actually processes an inputstream and sends output
 * to an outputstream. 
 */
void process_file(FILE *in, FILE *out)
{
    /* As long as we have not found the end of the file:
    */
    while(!feof(in))
    {
	/* locate the start string, don't 'print' anything yet
        */
        if (locate(in,out,options->bt,PRINT_NOTHING)) {
            
            /* found the start string, output the bytes defined:
            */
            putbuffer(out,options->pbbt);

            /* start string found: start 'printing' everything that 
             * follows, up to (and excluding) the token:
             */
            if (locate(in,out,options->et,PRINT_NON_TOKEN)) { 

                /* found the end token, output the bytes defined:
                */
                putbuffer(out,options->paet);
            } 
        }
    }
}

/* Find a sequence of characters in the input stream,
 */
int locate(FILE *instream, FILE *outstream, BYTEBUFFER what, unsigned int flags)
{
    int i=0,c;
    char *buffer;

    buffer=(char *)calloc(1,(size_t)what.bytecount);

    while( (c=fgetc(instream)) != EOF) 
    {

	if ((char)c==what.bytes[i]) 
        { 
            buffer[i]=c;
            i++; 

        } else { 

            flush_buffer(outstream,buffer,flags,&i);
            output(outstream,c,flags);	
        }

	if (i==what.bytecount) 
        { 
                flush_buffer(outstream,buffer,flags|TOKEN,&i);
		free(buffer); 
                return(1); 
        } /* MATCH FOUND */

    }
    free(buffer); return(0);
}

void flush_buffer(FILE *outstream, char *buffer, unsigned int flags, int *i)
{
    int j;

    /* flush buffer (if any) and print last character 
    */
    for(j=0;j<(*i);j++) output(outstream,buffer[j],flags); 
    *i=0; 
}

/*
 * Depending on the flags: put the current byte
 */
void output(FILE *outstream,int c,unsigned int flags) 
{
    if (flags & TOKEN) 
    {
	if (flags & PRINT_TOKEN) fputc(c,outstream);
    } else {
	if (flags & PRINT_NON_TOKEN) fputc(c,outstream);
    }
}


/* output the contents of a BYTEBUFFER 
*/
int putbuffer(FILE *fp, BYTEBUFFER p) 
{
    int n;
    for(n=0; n<p.bytecount; n++) putc(p.bytes[n],fp);
    return(n);
}


void print_help(char *programname)
{
    /* print settings
    */
    printf("\nUsage: %s [option..[option]] [files]\n",programname);
    printf("\nOptions with mandatory parameter [default]:\n\n");
    printf("-T --beginbytes              no output until these bytes found [url=\"]\n");
    printf("-b --print-before-beginbytes output this when beginbytes found []\n");
    printf("-t --endbytes                output until this token is found  [\"]\n");
    printf("-a --print-after-endbytes    output this after endbytes found  [\\n]\n");
    printf("\nParameters may contain these escape sequences:\n\n");
    printf("\\(#)  the byte represented by the decimal number #\n");
    printf("       e.g. \\(15) (0x0f) or \\(255) (0xff)\n\n");
    printf("\\\\    literal slash              \\n    newline\n");
    printf("\\r    carriage return            \\f    form feed\n");
    printf("\\v    vertical tab               \\b    backspace\n");
    printf("\\0    NULL character\n");
    printf("\nOptions that print information and exit:\n\n");
    printf("-h --help                    print this helpmessage\n");
    printf("-V --version                 print the version number\n");
    exit(0);
}

/*
 * eof()
 */
